/* CodeBox
Turns HTML textareas and inputs into code editors (syntax coloring, parenthesis matching, etc).

(c) Ice Fractal 2022 - www.icefractal.com
This code is open-source under the MIT License.
You must include this copyright notice in anything that includes this code or modified parts of it.

*/

/* 

CodeBox is compatible with existing code that uses a plain text element. The text is actually
still there in the textarea or input, however CodeBox has actually made the text invisible
and put the colored text in a div underneath! The only incompatibility is that the access
needs to go through the functions getCode() and setCode(). CodeBox moves the textarea's
classes and id to the parent div, which of course does not have a ".value" property.
Also, setCode() will automatically refresh the colored text.

Usage:

Simply add the class "codebox" to a textarea or input, and it will automatically be converted when the DOM loads.
You can optionally add a data-syntax attribute to specify what language syntax is used.

<textarea class="codebox" data-syntax="js"></textarea>

<input class="codebox" data-syntax="js"></input>

Add the class "dark" to use a dark background with light text for the syntax coloring:

<textarea class="codebox dark" data-syntax="glsl"></textarea>

Add the class "linenums" to show code line numbers on the left edge:

<textarea class="codebox linenums" data-syntax="js"></textarea>

*/


// Returns the textarea or input element that contains the code.
function getCodeBox(codebox) {if (codebox.value !== undefined) return codebox; else return codebox.firstChild.firstChild.lastChild;}

// Gets the code string in a CodeBox.
function getCode(codebox) {return getCodeBox(codebox).value;}

// Sets the code string in a CodeBox.
function setCode(codebox, code) {if (codebox.value !== undefined) codebox.value=code; else {var i = codebox.firstChild.firstChild.lastChild; i.value = code; _CodeBoxInternal.oninput(i, true);}}

// Refreshes the formatting in a CodeBox.
function refreshCodeBox(codebox) {if (codebox.value === undefined) {var i = codebox.firstChild.firstChild.lastChild; _CodeBoxInternal.oninput(i, true);}}

// Sets the syntax highlighting mode in a CodeBox.
function setCodeSyntax(codebox, syntax) {getCodeBox(codebox).setAttribute("data-syntax", syntax); refreshCodeBox(codebox);}

// Scans the document for codeboxes and initializes them.
function initCodeBoxes() {return _CodeBoxInternal.initCodeBoxes();}

// ----

// Internal code for handling CodeBox functions.
var _CodeBoxInternal = new function() {

this.initCodeBoxes = function() {
	var boxes = document.body.getElementsByClassName("codebox");

	var bxs = [];
	for (var c = 0; c < boxes.length; c++)
		bxs.push(boxes[c]);
	
	for (var c = 0; c < bxs.length; c++) {
		var tag = bxs[c].tagName.toLowerCase();
		var isinput = tag == "input";
		if (!isinput && tag != 'textarea') continue;
		
		var outer = document.createElement(isinput ? "span" : "div");
		
		var showlinenums = bxs[c].classList.contains("linenums");
		
		if (showlinenums)
			outer.innerHTML = "<div><div><span></span><div></div></div></div>";
		else
			outer.innerHTML = "<div><div><div></div></div></div>";
		
		var pad = window.getComputedStyle(bxs[c]);
		if (pad) pad = pad['padding-top'] + ' ' + pad['padding-right'] + ' ' + pad['padding-bottom'] + ' ' + pad['padding-left'];
		else pad = "2px";
		
		bxs[c].parentNode.insertBefore(outer, bxs[c]);
		outer.firstChild.firstChild.appendChild(bxs[c]);
		
		outer.className = bxs[c].className;
		bxs[c].removeAttribute("class");
		
		if (bxs[c].id) {
			outer.id = bxs[c].id;
			bxs[c].removeAttribute("id");
		}
		
		if (bxs[c].getAttribute("style")) {
			outer.setAttribute("style", bxs[c].getAttribute("style"));
			bxs[c].removeAttribute("style");
		}
		
		bxs[c].setAttribute("spellcheck", "false");
		bxs[c].setAttribute("autocorrect", "off");
		bxs[c].setAttribute("autocapitalize", "off");
		
		var itext = bxs[c];
		var ctext = itext.previousSibling;
		
		if (!isinput) {
			itext.style.padding = pad;
			ctext.style.padding = pad;
			if (showlinenums) ctext.previousSibling.style.padding = pad;
		}
		bxs[c].addEventListener("input", codebox_oninput);
		bxs[c].addEventListener("mousedown", function() {codebox_arefpars(this);});
		bxs[c].addEventListener("touchend", function() {codebox_arefpars(this);});
		bxs[c].addEventListener("keydown", codebox_onkeydown);
		bxs[c].addEventListener("blur", function() {clearPars(this);});
		
		codebox_oninput.call(bxs[c], true);
	}
	
	if (bxs.length > 0 && document.head.getElementsByClassName("codeboxstyle").length == 0) {
		var style = document.createElement("style");
		style.className = "codeboxstyle";
		
		style.innerHTML = `
			.codebox {
				padding: 0px !important;
				box-sizing: border-box;
			}
			.codebox, .codebox * {
				font-family: 'Lucida Console', monospace;
				font-size: 10pt;
				-webkit-text-size-adjust: none;
			}
			.codebox > div {
				position: relative;
				overflow-x: auto;
				overflow-y: hidden;
				box-sizing: border-box;
			}

			div.codebox > div {height: 100%; overflow: auto;}

			.codebox > div > div {
				position: relative;
				display: inline-block;
				min-width: 100%;
				min-height: 100%;
			}

			.codebox div div div, .codebox textarea, .codebox input {
				white-space: pre;
				overflow: hidden;
				resize: none;
				border: 0 !important;
				outline: none !important;
				box-shadow: none !important;
				box-sizing: border-box;
				padding: 0.1em;
				margin: 0;
				display:block;
				line-height:1.1em;
				-moz-tab-size : 4;
				-o-tab-size : 4;
				tab-size : 4;
			}
			.codebox textarea, .codebox input {
				position: absolute;
				left: 0;
				top: 0;
				width: 100%;
				height: 100%;
				
				color: #000;
				-webkit-text-fill-color: transparent;//rgba(255,0,0,128);
				caret-color: #000;
				background: transparent;
				//display:none;
			}
			.codebox div div div {
				pointer-events: none;
				user-select: none;
				-webkit-user-select: none;
				min-height: 1.1em;
				min-width: 10px;
			}
	
			span.codebox {
				height: 1.5em !important;
				overflow:hidden;
				display: inline-block;
			}
			span.codebox div div div, .codebox input {
				line-height: 1.2em;
				height: auto;
			}
			.codebox .selpar {color: blue; text-shadow: 0 0 1px blue;}
			.codebox .selparbroken {color: red; text-shadow: 0 0 1px red;}
			
			.codebox.dark .selpar {color: #08F; text-shadow: 0 0 1px blue, 0 0 2px blue;}
			
			div.codebox, span.codebox {background: #FFF; color: #000;}
			div.codebox.dark, span.codebox.dark {background: #000; color: #FFF;}
			
			.codebox.dark textarea, .codebox.dark input {
				color: #FFF;
				caret-color: #FFF;
			}
			
			.codebox.linenums > div > div {
				margin-left: 45px;
				min-width: calc(50% - 45px);
			}
			
			.codebox.linenums > div > div > span {
				position: absolute;
				left: -45px;
				top: 0;
				min-height: 100%;
				width: 45px;
				background: linear-gradient(to right, #7799AA50, #7799AA20);
				border-right: 1px solid #8883;
				box-sizing: border-box;
				display: block;
				white-space: pre;
				line-height:1.1em;
				text-align: right;
				color: #444;
			}
			.codebox.dark.linenums > div > div > span {
				color: #CCC;
			}
			.cssimprt {color:red;text-shadow: 0 0 1px #FF8000;}
			.phptag {color:#00F;background:#AFF;outline:1px solid #AFF; text-shadow: 0 0 1px #0FF;}
		`;
		
		document.head.appendChild(style);
		
		document.addEventListener("selectionchange", function(e) {
			if (document.activeElement.tagName.toLowerCase() == 'textarea' && document.activeElement.parentNode.className.indexOf("codebox") != -1)
				refreshPars(document.activeElement);
		});
	}
	
}

window.addEventListener("load", initCodeBoxes);

function codebox_arefpars(itext) {
	setTimeout(function() {refreshPars(itext);}, 10);
}

function codebox_onkeydown(e) {
	codebox_arefpars(this);
	console.log(e);
	if (e.keyCode == 9) { // TAB
		var start = this.selectionStart;
		var end = this.selectionEnd;
		
		var seltext = this.value.substring(start, end);
		
		if (e.shiftKey || seltext != "") { // Multi-line selection, so change the indentation.
			while (start > 0 && this.value[start-1] != '\n') start--;
			while (this.value[end] && this.value[end] != '\n') end++;
			
			var lines = this.value.substring(start, end).split("\n");
			var delta = 0;
			for (var c = 0; c < lines.length; c++) {
				if (e.shiftKey) {
					if (lines[c][0] == '\t') {
						lines[c] = lines[c].substr(1);
						delta--;
					}
				} else {
					lines[c] = '\t' + lines[c];
					delta++;
				}
			}
			
			this.value = this.value.substr(0, start) + lines.join("\n") + this.value.substr(end);
			this.selectionStart = start;
			this.selectionEnd = end + delta;
		} else {
			this.value = this.value.substr(0, start) + "\t" + this.value.substr(this.selectionEnd);
			this.selectionStart = this.selectionEnd = start+1;
		}
		
		e.preventDefault();
		codebox_oninput.call(this);
	} else if (e.keyCode == 10 || e.keyCode == 13) { // Enter
		var start = this.selectionStart;
		
		var tabs = '';
		var ix = start;
		while (ix > 0 && this.value[ix-1] != '\n') ix--;
		while (this.value[ix] == '\t') {tabs += '\t'; ix++}
		
		this.value = this.value.substr(0, this.selectionStart) + "\n" + tabs + this.value.substr(this.selectionEnd);
		this.selectionStart = this.selectionEnd = start+1+tabs.length;
		e.preventDefault();
		codebox_oninput.call(this);
	}
}

var TextColorizer = function() {
	this.html = "";
	this.color = "";
	
	var dummy = document.createElement("div");
	
	this.setColor = function(color) {
		if (this.color != color) {
			if (this.color) {this.html += "</span>";}
			this.color = color;
			if (color) {this.html += "<span style='color:#"+color+";'>";}
		}
	}
	this.finish = function() {this.setColor(""); return this.html;}
	
	this.appendText = function(text) {
		dummy.innerText = text;
		this.html += dummy.innerHTML;
	}
	this.appendColoredText = function(text, color) {
		this.setColor(color);
		this.appendText(text);
		this.setColor();
	}
	this.appendClassedText = function(text, className) {
		if (this.color) {this.html += "</span>"; this.color = "";}
		this.html += "<span class='"+className+"'>" + text + "</span>";
	}
}

// JS
var js_keywords = ["=>","true","false","async","await","break","case","catch","class","const","continue","debugger","default","delete","do","else","export","extends","finally","for","function","if","import","in","instanceof","let","new","null","of","return","super","switch","this","throw","try","typeof","undefined","var","void","while","with","yield"];

// PHP
var php_keywords = ["true","false","abstract","and","array","as","break","callable","case","catch","class","clone","const","continue","declare","default","die","do","echo","else","elseif","empty","enddeclare","endfor","endforeach","endif","end","switch","end","while","eval","exit","extends","final","finally","fn","for","foreach","function","global","goto","if","implements","include","include_once","instanceof","insteadof","interface","isset","list","match","namespace","new","or","print","private","protected","public","readonly","require","require_once","return","static","switch","throw","trait","try","unset","use","var","while","xor"];

// GLSL
var glsl_keywords = ["attribute","const","uniform","varying","layout","centroid","flat","smooth","noperspective","patch","sample","break","continue","do","for","while","switch","case","default","if","else","subroutine","in","out","inout","float","double","int","void","bool","true","false","invariant","discard","return","mat2","mat3","mat4","dmat2","dmat3","dmat4","mat2x2","mat2x3","mat2x4","dmat2x2","dmat2x3","dmat2x4","mat3x2","mat3x3","mat3x4","dmat3x2","dmat3x3","dmat3x4","mat4x2","mat4x3","mat4x4","dmat4x2","dmat4x3","dmat4x4","vec2","vec3","vec4","ivec2","ivec3","ivec4","bvec2","bvec3","bvec4","dvec2","dvec3","dvec4","uint","uvec2","uvec3","uvec4","lowp","mediump","highp","precision","sampler1D","sampler2D","sampler3D","samplerCube","sampler1DShadow","sampler2DShadow","samplerCubeShadow","sampler1DArray","sampler2DArray","sampler1DArrayShadow","sampler2DArrayShadow","isampler1D","isampler2D","isampler3D","isamplerCube","isampler1DArray","isampler2DArray","usampler1D","usampler2D","usampler3D","usamplerCube","usampler1DArray","usampler2DArray","sampler2DRect","sampler2DRectShadow","isampler2DRect","usampler2DRect","samplerBuffer","isamplerBuffer","usamplerBuffer","sampler2DMS","isampler2DMS","usampler2DMS","sampler2DMSArray","isampler2DMSArray","usampler2DMSArraysamplerCubeArray","samplerCubeArrayShadow","isamplerCubeArray","usamplerCubeArraystruct"];

/*
Built-ins:
Inputs:

gl_InstanceID
gl_DrawID
gl_BaseVertex
gl_BaseInstance



gl_PointCoord

Outputs:
gl_VertexID
gl_PointSize
gl_ClipDistance
gl_FragDepth


*/
var glsl_builtins = ["gl_Position","gl_PointSize","gl_FragCoord","gl_FrontFacing","gl_FragColor","gl_FragData","gl_MaxVertexAttribs","gl_MaxVertexUniformVectors","gl_MaxVaryingVectors","gl_MaxVertexTextureImageUnits","gl_MaxCombinedTextureImageUnits","gl_MaxFragmentUniformVectors","gl_MaxDrawBuffers"];

function highlightSyntax(t, syntax, dark) {
	const letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_$';
	const digits = '0123456789';
	const digitsnd = '.0123456789';
	const digitsndx = '.0123456789abcdefxonABCDEFXON';
	const whitespace = ' \t\n\r';

	const SYNTAX_MATH = 0;
	const SYNTAX_HTML = 1;
	const SYNTAX_CSS = 2;
	const SYNTAX_PHP = 3;
	const SYNTAX_GLSL = 4;
	const SYNTAX_JS = 5;

	var codebox_colors = {
		normal: "",
		func: "07B",
		keyword: "00F",
		number: "007F7F",
		string: "7F007F",
		comment: "007F00",
		preproc: "7F7F00",
		htmltag: "000090",
		phpvar: "0000A0",
	};

	var codebox_darkcolors = {
		normal: "C0E8FF",
		func: "9EF",
		keyword: "50B8FF",
		number: "00F0F0",
		string: "F0F",
		comment: "40D040",
		preproc: "C8A800",
		htmltag: "50B8FF",
		phpvar: "83B8F5",
	};
	
	var tc = new TextColorizer();
	
	var c = 0;
	
	var colors = dark ? codebox_darkcolors : codebox_colors;
	
	var context = null;
	
	var sx;
	if (syntax == "math")
		sx = SYNTAX_MATH;
	else if (syntax == "html")
		sx = SYNTAX_HTML;
	else if (syntax == "css")
		sx = SYNTAX_CSS;
	else if (syntax == "glsl")
		sx = SYNTAX_GLSL;
	else if (syntax == "php")
		sx = SYNTAX_HTML;
	else sx = SYNTAX_JS;

langloop:
	while (c < t.length) {
	
	// TODO: Implement dark mode for HTML, CSS, and PHP syntax.
	
	if (sx == SYNTAX_MATH) {
		var level = 0;
		var lix = 0;
		while (c < t.length) {
			if (t[c] == '(') {
				tc.appendColoredText(t.substring(lix, c), colors.func);
				lix = c+1;
				tc.appendClassedText(t[c], 'leftroundpar par c' + c + ' parlevel' + (level%3));
				level++;
			} else if (t[c] == ')') {
				tc.appendText(t.substring(lix, c));
				lix = c+1;
				level--;
				tc.appendClassedText(t[c], 'riteroundpar par c' + c + ' parlevel' + (level%3));
			} else {
				if (c+1 == t.length || letters.indexOf(t[c]) == -1 && digitsnd.indexOf(t[c]) == -1 && whitespace.indexOf(t[c]) == -1) {
					tc.appendText(t.substring(lix, c+1));
					lix = c+1;
				}
			}
			c++;
		}
	} else if (sx == SYNTAX_HTML) {
		var inelem = false;
		var inx = false;
		var tagix = "";
		while (c < t.length) { // TODO: Add comments.
			if (t.substr(c, 4) == '<!--') {
				tc.setColor(colors.comment);
				do {
					tc.appendText(t[c++]);
				} while (c < t.length && t.substr(c-3, 3) != '-->' && t.substr(c-4, 4) != '--!>');
				
				tc.setColor("");
				continue;
			}
			if (t[c] == '<' && t[c+1] == '!') {
				tc.setColor("7070FF");
				tagix = c+1;
				inx = inelem = true;
			} else if (t[c] == '<') {
				if (syntax == "php" && t.substr(c+1, 4) == "?php") {context = sx; sx = SYNTAX_PHP; continue langloop;}
				tc.setColor(colors.htmltag);
				tagix = c+1;
				inelem = true;
			} else if (t[c] == '>') {
				tc.setColor(inx ? "7070FF" : colors.htmltag);
				tc.appendText('>');
				tc.setColor("");
				inelem = inx = false;
				c++;
				var tag = '';
				while (t[tagix] && t[tagix] != ' ' && t[tagix] != '\n' && t[tagix] != '>') tag += t[tagix++];
				tag = tag.toLowerCase();
				//console.log("tag = " + tag);
				if (tag == "script") {context = sx; sx = SYNTAX_JS; continue langloop;}
				if (tag == "style") {context = sx; sx = SYNTAX_CSS; continue langloop;}
				continue;
			}
			
			if (inelem && whitespace.indexOf(t[c]) != -1) {
				tc.setColor(colors.number);
			}
			if (inelem && t[c] == '=') {
				tc.setColor(colors.htmltag);
				tc.appendText('=');
				tc.setColor(colors.string);
				c++;
				continue;
			}
			if (inelem && (t[c] == '"' || t[c] == '\'')) {
				var d = t[c];
				var e = c;
				do {e++} while (e < t.length-1 && t[e] != d);
				tc.setColor(colors.string);
				tc.appendText(t.substring(c, c = e+1));
				tc.setColor(colors.number);
				continue;
			}
			
			tc.appendText(t[c]);
			
			c++;
		}
	} else if (sx == SYNTAX_CSS) { // NEEDS WORK. :hover, @import, @font, etc.
		var inelem = false;
		var inquery = false;
		while (c < t.length) {
			if (context && t.substr(c, 8).toLowerCase() == "</style>") {sx = context; context = null; continue langloop;}
			if (t[c] == '/' && t[c+1] == '*') {
				tc.setColor(colors.comment);
				do {
					tc.appendText(t[c++]);
				} while (c < t.length && (t[c-2] != '*' || t[c-1] != '/'));
				
				tc.setColor("");
			} else if (t[c] == '/' && t[c+1] == '/') {
				tc.setColor(colors.comment);
				do {
					tc.appendText(t[c++]);
				} while (c < t.length && (t[c-1] != ';'));
				
				tc.setColor("");
			} else {
				if (t.substr(c, 10).toLowerCase() == '!important') {
					tc.appendClassedText(t.substr(c, 10), 'cssimprt');
					c += 10;
					continue;
				}
				if (!inelem) tc.setColor(colors.keyword);
				if (t[c] == '@') {
					inquery = true;
					tc.setColor(colors.preproc);
				}
				if (!inquery && t[c] == ':') {
					tc.setColor("");
					tc.appendText(':');
					tc.setColor(colors.string);
				} else if (t[c] == ';') {
					inquery = false;
					tc.setColor("");
					tc.appendText(';');
					tc.setColor(colors.normal);
				} else if (t[c] == '{') {
					tc.appendClassedText(t[c], 'leftbracepar par c' + c);
					if (inquery) {
						inquery = false;
					} else {
						inelem = true;
						tc.setColor(colors.normal);
					}
				} else if (t[c] == '}') {
					inelem = false;
					tc.appendClassedText(t[c], 'ritebracepar par c' + c);
				} else
					tc.appendText(t[c]);
					
				c++;
			}
		}
	} else {
		var keywords = sx == SYNTAX_GLSL ? glsl_keywords : (sx == SYNTAX_PHP ? php_keywords : js_keywords);
		outer:
		while (c < t.length) {
			if (context && t.substr(c, 9).toLowerCase() == "</script>") {sx = context; context = null; continue langloop;}
			for (var k = 0; k < keywords.length; k++) {
				if (c + keywords[k].length <= t.length && letters.indexOf(t[c-1]) == -1 && letters.indexOf(t[c+keywords[k].length]) == -1 && t.substr(c, keywords[k].length) == keywords[k]) {
					tc.appendColoredText(keywords[k], colors.keyword);
					c += keywords[k].length;
					continue outer;
				}
			}
			if (sx == SYNTAX_GLSL) {
				for (var k = 0; k < glsl_builtins.length; k++) {
					if (c + glsl_builtins[k].length <= t.length && letters.indexOf(t[c-1]) == -1 && letters.indexOf(t[c+glsl_builtins[k].length]) == -1 && t.substr(c, glsl_builtins[k].length) == glsl_builtins[k]) {
						tc.appendColoredText(glsl_builtins[k], colors.keyword);
						c += glsl_builtins[k].length;
						continue outer;
					}
				}
			} else if (sx == SYNTAX_PHP) {
				if (t.substr(c, 5) == '<?php') {
					tc.appendClassedText('&lt;?php', "phptag");
					c += 5;
					continue;
				} else if (t.substr(c, 2) == '?>') {
					tc.appendClassedText('?&gt;', "phptag");
					c += 2;
					sx = SYNTAX_HTML;
					continue langloop;
				}
			}
			if (digits.indexOf(t[c]) != -1 && letters.indexOf(t[c-1]) == -1 && (t[c] != '.' || digitsnd.indexOf(t[c+1]) != -1)) {
				tc.setColor(colors.number);
				
				do {
					tc.appendText(t[c]);
					c++;
					if (c >= t.length) break;
					var lc = t[c].toLowerCase();
				} while (digitsndx.indexOf(t[c]) != -1);
				
				
				tc.setColor("");
			} else if (t[c] == '"' || t[c] == "'" || t[c] == "`") {
				tc.setColor(colors.string);
				var d = t[c];
				tc.appendText(d);
				do {
					c++;
					if (c >= t.length) break;
					tc.appendText(t[c]);
				} while ((t[c] != d || (t[c-1] == '\\' && t[c-2] != '\\')));
				c++;
				
				tc.setColor("");
			} else if (t[c] == '/' && t[c+1] == '/') {
				tc.setColor(colors.comment);
				while (c < t.length && t[c] != '\n') {
					if (sx == SYNTAX_PHP && t.substr(c,2) == "?>") continue langloop;
					tc.appendText(t[c++]);
				}
				tc.setColor("");
			} else if (t[c] == '#') {
				tc.setColor(colors.preproc);
				while (c < t.length && t[c] != '\n')
					tc.appendText(t[c++]);
				tc.setColor("");
			} else if (sx == SYNTAX_PHP && t[c] == '$') {
				tc.setColor(colors.phpvar);
				while (c < t.length && letters.indexOf(t[c]) != -1)
					tc.appendText(t[c++]);
				tc.setColor("");
			} else if (t[c] == '/' && t[c+1] == '*') {
				tc.setColor(colors.comment);
				do {
					tc.appendText(t[c++]);
				} while (c < t.length && (t[c-2] != '*' || t[c-1] != '/'));
				
				tc.setColor("");
			} else {
				if (t[c] == '(')
					tc.appendClassedText(t[c], 'leftroundpar par c' + c);
				else if (t[c] == '{')
					tc.appendClassedText(t[c], 'leftbracepar par c' + c);
				else if (t[c] == '[')
					tc.appendClassedText(t[c], 'leftbracketpar par c' + c);
				else if (t[c] == ')')
					tc.appendClassedText(t[c], 'riteroundpar par c' + c);
				else if (t[c] == '}')
					tc.appendClassedText(t[c], 'ritebracepar par c' + c);
				else if (t[c] == ']')
					tc.appendClassedText(t[c], 'ritebracketpar par c' + c);
				else {
					var isletter = letters.indexOf(t[c]) != -1;
					tc.setColor(isletter ? colors.normal : "");
					do {
						tc.appendText(t[c]);
						c++;
					} while (isletter && t[c] && digits.indexOf(t[c]) != -1);
					tc.setColor("");
					continue;
				}
				c++;
			}
		}
	}
	
	}
	return tc.finish();
}

function codebox_oninput(skippars) {
	var itext = this;
	var ctext = this.previousSibling;
	var codebox = itext.parentNode.parentNode.parentNode;
	var t = itext.value;
	
	itext.style['-webkit-text-fill-color'] = t == '' ? 'black' : 'transparent';
	
	var syntax = itext.getAttribute("data-syntax");
	if (syntax) syntax = syntax.toLowerCase();
	
	if (codebox.classList.contains("linenums")) {
		var numbox = codebox.firstChild.firstChild.firstChild;
		var s = "";
		var lines = t.split("\n").length;
		for (var c = 1; c <= lines; c++) {
			s += c + "\n";
		}
		numbox.innerHTML = s;
	}
	
	ctext.innerHTML = highlightSyntax(t, syntax, codebox.classList.contains("dark"));
	
	var pars = ctext.getElementsByClassName("par");
	for (var c = 0; c < pars.length; c++) {
		if (!findParMatch(pars[c]))
			pars[c].className += " selparbroken";
	}
	
	function resetscroll() {
		itext.scrollTop = 0;
		itext.scrollLeft = 0;
	}
	setTimeout(resetscroll, 100);
	setTimeout(resetscroll, 700);
	
	if (skippars !== true) refreshPars(itext);
}

function findParMatch(par) {
	var dir = par.className.indexOf("left") == -1 ? 'rite' : 'left';
	var next = dir == 'rite' ? "previousSibling" : "nextSibling";
	var type = par.className.substring(4, par.className.indexOf('par')+3);
	
	var level = 0;
	while (par = par[next]) {
		if ((par.className+'').indexOf(type) == -1) continue;
		
		if (par.className.indexOf(dir + type) == -1) {
			if (level)
				level--;
			else
				return par;
		} else
			level++;
	}
	return null;
}

function clearPars(itext) {
	var sp = itext.previousSibling.getElementsByClassName("selpar");
	
	while (sp.length)
		sp[0].className = sp[0].className.replace(" selpar", "");
}

function refreshPars(itext) {
	//if (itext.getAttribute("data-syntax") == 'math') return;
	
	var ctext = itext.previousSibling;
	
	var i = itext.selectionStart;
	if (i != itext.selectionEnd) return;
	var t = itext.value;
	
	clearPars(itext);
	
	var par = ctext.getElementsByClassName("c" + (i-1));
	if (!par.length) par = ctext.getElementsByClassName("c" + i);
	if (!par.length) return;
	par = par[0];
	
	var par2 = findParMatch(par);
	
	if (par2) {
		par.className += ' selpar';
		par2.className += ' selpar';
	}
	
}

this.oninput = function(codebox, skippars) {codebox_oninput.call(codebox, skippars);}

};
