// Color Dialog
// (c) Ice Fractal 2019 - www.icefractal.com
// 
// This code is open-source under the MIT License.
//
// You must include this copyright notice in anything that includes this code or modified parts of it.
//

// Requires no external JS libraries.

/* ---------- Example ---------------

var cd = new ColorDialog();

cd.show(mybutton_id_string);

cd.onclose = function(css_color) {
	alert(css_color);
};

*/

/* ---------- Functions ---------------

// ----------------------------
// Constructor - Constructs a dialog object.
//
// parent_elem	- [Optional] Either a string ID or a DOM element object. The dialog will be displayed beneath that element on the screen.
//
// Returns		- A ColorDialog object.

new ColorDialog(parent_elem);

// ----------------------------
// Function - Shows the dialog.
//
// parent_elem	- [Optional] Either a string ID or a DOM element object. The dialog will be displayed beneath that element on the screen.
//
// Returns		- None

cd.show(parent_elem);

// ----------------------------
// Function - Closes (hides) the dialog.
//

cd.hide();

// ----------------------------
// Function - Gets the currently selected color.
//
// Returns		- A CSS color string of the currently selected color.

cd.getColor(parent_elem);

// ----------------------------
// Function - Gets the currently selected color.
//
// Returns		- An RGBA array of color components in this order: [R, G, B, A]

cd.getColorRGBA(parent_elem);

// ----------------------------
// Function - Sets the currently selected color.
//
// color	- Either an array of RGBA components or a CSS "rgba(#, #, #, #)" format string.
//
// Returns		- None

cd.setColor(parent_elem);

// ----------------------------
// Function - Checks if the dialog is currently open.
//
// Returns		- true only if the dialog is currently open.

cd.isShown();

// ----------------------------
// Event - Called when the dialog is closed.
//
// css_color	- A CSS color string of the currently selected color.
//

cd.onclose = function(css_color) {};

// ----------------------------
// Event - Called when the dialog is closed.
//
// css_color	- A CSS color string of the currently selected color.
//

cd.onchange = function(css_color) {};

*/

var ColorDialog = function(parent_elem) {
	
	// Inject the SVG code directly to avoid all the JavaScript namespace complications.
	var SHEET_SVGCODE = '<svg viewBox="0 0 131 125" xmlns="http://www.w3.org/2000/svg" style="width:240px;height:240px;"><defs><clipPath id="sheetclip"><rect x="0" y="0" width="100" height="100"></rect></clipPath><linearGradient id="huegrad" x1="0%" y1="0%" x2="100%" y2="0%"><stop offset="0%" style="stop-color:rgb(255,0,0);" /><stop offset="16.7%" style="stop-color:rgb(255,255,0);" /><stop offset="33.3%" style="stop-color:rgb(0,255,0);" /><stop offset="50%" style="stop-color:rgb(0,255,255);" /><stop offset="66.7%" style="stop-color:rgb(0,0,255);" /><stop offset="83.3%" style="stop-color:rgb(255,0,255);" /><stop offset="100%" style="stop-color:rgb(255,0,0);" /></linearGradient><linearGradient id="satgrad" x1="0%" y1="0%" x2="0%" y2="100%"><stop offset="0%" style="stop-color:rgba(128,128,128,0);" /><stop offset="100%" style="stop-color:rgb(128,128,128,255);" /></linearGradient><linearGradient id="lumgrad" x1="0%" y1="0%" x2="0%" y2="100%"><stop offset="0%" style="stop-color:rgb(255,255,255);" /><stop class="cd_midlumstop" offset="50%" style="stop-color:rgb(128,128,128);" /><stop offset="100%" style="stop-color:rgb(0,0,0);" /></linearGradient><linearGradient id="alphagrad" x1="0%" y1="0%" x2="0%" y2="100%"><stop class="cd_fullalphastop" offset="0%" style="stop-color:rgba(255,255,255,255);" /><stop offset="100%" class="cd_zeroalphastop" style="stop-color:rgba(255,255,255,0);" /></linearGradient><pattern id="alphapat" x="0" y="0" width="1" height=".1"><rect x="0" y="0" width="5" height="5" fill="white"></rect><rect x="0" y="5" width="5" height="5" fill="#bbb"></rect><rect x="5" y="0" width="5" height="5" fill="#bbb"></rect><rect x="5" y="5" width="5" height="5" fill="white"></rect></pattern><pattern id="cprevpat" x="0" y="0" width="0.077" height=".5"><rect x="0" y="0" width="5" height="5" fill="white"></rect><rect x="0" y="5" width="5" height="5" fill="#bbb"></rect><rect x="5" y="0" width="5" height="5" fill="#bbb"></rect><rect x="5" y="5" width="5" height="5" fill="white"></rect></pattern></defs><g class="cd_maing"><rect x="0" y="0" width="1000" height="1000" stroke="transparent" fill="transparent"></rect><rect x="0" y="105" width="131" height="20" fill="url(#cprevpat)"></rect><rect class="cd_colorprev" x="0" y="105" width="131" height="20"></rect><rect x="0" y="0" width="100" height="100" fill="url(#huegrad)"></rect><rect x="0" y="0" width="100" height="100" fill="url(#satgrad)"></rect><rect x="105" y="0" width="10" height="100" fill="url(#lumgrad)"></rect><rect x="120" y="0" width="10" height="100" fill="url(#alphapat)"></rect><rect x="120" y="0" width="10" height="100" fill="url(#alphagrad)"></rect><g class="cd_lumhandle" transform="translate(0,50)"><rect x="104" y="-1.5" width="12" height="3" fill="rgba(0,0,0,0.35)" stroke="white"></rect></g><g class="cd_alphahandle" transform="translate(0,50)"><rect x="119" y="-1.5" width="12" height="3" fill="rgba(0,0,0,0.35)" stroke="white"></rect></g><g clip-path="url(#sheetclip)"><g class="cd_huehandle" transform="translate(50,50)"><polygon points="-2,-8 2,-8 0,-2" style="fill:rgba(0,0,0,0.5);stroke:rgba(255,255,255,0.85);"></polygon><polygon points="8,-2 8,2 2,0" style="fill:rgba(0,0,0,0.5);stroke:rgba(255,255,255,0.85);"></polygon><polygon points="-8,-2 -8,2 -2,0" style="fill:rgba(0,0,0,0.5);stroke:rgba(255,255,255,0.85);""></polygon><polygon points="-2,8 2,8 0,2" style="fill:rgba(0,0,0,0.5);stroke:rgba(255,255,255,0.85);"></polygon></g></g></g></svg>';
	
	this.parent = (typeof(parent_elem) == "string") ? document.getElementById(parent_elem) : parent_elem;
	
	this.onchange = function() {}
	this.onopen = function() {}
	this.onclose = function() {}
	
	var that = this;
	
	var curcolor = [80,150,100,1];
	
	function _HSL_comp_To_RGB(t, x, y) {
		if(t < 0.0) t += 1.0;
		if(t > 1.0) t -= 1.0;
		if(t < 1.0/6.0) return t * (y - x) * 6.0 + x;
		if(t < 0.5) return y;
		if(t < 2.0/3.0) return (2.0/3.0 - t) * (y - x) * 6.0 + x;
		return x;
	}

	function HSL_To_RGB(h, s, l) {
		var rgb = [0,0,0];

		if (s == 0)
			rgb[0] = rgb[1] = rgb[2] = l;
		else {
			var y = (l < 0.5) ? ((1.0 + s) * l) : (s + l - l*s);
			var x = 2.0 * l - y;
			rgb[0] = _HSL_comp_To_RGB(h + 1.0/3.0, x, y);
			rgb[1] = _HSL_comp_To_RGB(h, x, y);
			rgb[2] = _HSL_comp_To_RGB(h - 1.0/3.0, x, y);
		}

		rgb[0] = clip(Math.round(rgb[0] * 255), 0, 255);
		rgb[1] = clip(Math.round(rgb[1] * 255), 0, 255);
		rgb[2] = clip(Math.round(rgb[2] * 255), 0, 255);

		return rgb;
	}

	function RGB_To_HSL(r, g, b) {
		r /= 255.0;
		g /= 255.0;
		b /= 255.0;

		var min = Math.min(r, g, b), max = Math.max(r, g, b);
		var esum = max + min;

		var hsl = [0, 0, esum * 0.5];
		
		if (max == min)
			return hsl;
		else {
			var d = max - min;

			hsl[1] = d / (esum < 1.0 ? esum : (2.0 - esum));

			if (max == r)
				hsl[0] = (g - b) / d + (g < b ? 6.0 : 0.0);
			else if (max == g)
				hsl[0] = (b - r) / d + 2.0;
			else
				hsl[0] = (r - g) / d + 4.0;

			hsl[0] /= 6.0;
		}

		return hsl;
	}

	var DRAG_NONE = 0;
	var DRAG_SHEET = 1;
	var DRAG_LUMBAR = 2;
	var DRAG_ALPHABAR = 3;

	var dragmode = false;

	function PtToSVG(x, y) {
		var pt = that.svg.createSVGPoint();
		pt.x = x; pt.y = y;
		
		var sp = pt.matrixTransform(that.svg.getScreenCTM().inverse());
		return [sp.x, sp.y];
	}

	function clip(n, min, max) {
		return (n < min) ? min : ((n > max) ? max : n);
	}
	
	function hex00(n) {
		n = (+n).toString(16).toUpperCase();
		return (n.length == 1) ? ("0"+n) : n;
	}
	
	function RGBA_To_CSS(rgba) {
		return "#" + hex00(rgba[0]) + hex00(rgba[1]) + hex00(rgba[2]) + hex00(Math.round(rgba[3]*255.0));
		//return "rgba(" + rgba[0] + "," + rgba[1] + "," + rgba[2] + "," + rgba[3] + ")";
	}

	function CSS_To_RGBA(css_color) {
		if (css_color[0] == '#') {
			var hex = css_color;
			console.log("hex = " + hex);
			css_color = "rgba(" + parseInt(hex.substr(1,2), 16) + ',' + parseInt(hex.substr(3,2), 16) + ',' + parseInt(hex.substr(5,2), 16);
			if (hex.length > 7) css_color += ',' + (Math.round(parseInt(hex.substr(7,2), 16) / 2.55) / 100.0);
			css_color += ')';
			console.log("css_color = " + css_color);
		}
		if (css_color.indexOf("rgba(") == -1) return [0,0,0,1];
		var rgba = css_color.replace("rgba(", "").replace(")", "").split(",");
		for (var c = 0; c < rgba.length; c++)
			rgba[c] = parseFloat(rgba[c]);
		return rgba;
	}
	this.CSS_To_RGBA = CSS_To_RGBA;

	function mouseDown(e) {
		var pt = PtToSVG(e.clientX, e.clientY);
		var x = pt[0], y = pt[1];
		
		if (x < 100)
			dragmode = DRAG_SHEET;
		else if (x > 105 && x < 115)
			dragmode = DRAG_LUMBAR;
		else if (x > 120)
			dragmode = DRAG_ALPHABAR;
		else
			dragmode = DRAG_NONE;
		
		if (dragmode != DRAG_NONE && e.preventDefault) e.preventDefault();
		
		console.log("dragmode = " + dragmode);
	}

	function mouseUp() {dragmode = DRAG_NONE;}

	function mouseMove(e) {
		if (that.cd_div === undefined) return;
		var pt = PtToSVG(e.clientX, e.clientY);
		pt[0] = clip(pt[0], 0, 100);
		pt[1] = clip(pt[1], 0, 100);
		console.log("dragmode = " + dragmode);
		switch (dragmode) {
			case DRAG_SHEET:
				that.huehandle.setAttribute("transform", "translate(" + pt[0] + "," + pt[1] + ")");
				break;
			case DRAG_LUMBAR:
				that.lumhandle.setAttribute("transform", "translate(0," + pt[1] + ")");
				break;
			case DRAG_ALPHABAR:
				that.alphahandle.setAttribute("transform", "translate(0," + pt[1] + ")");
				break;
			default: return;
		}
		;
		that.onchange(that.getColor());
	}
	
	function getTransform(xf) {
		var translate =  xf.replace(")", "").replace("translate(", "").split(",");
		return [parseFloat(translate[0]), parseFloat(translate[1])];
	}
	
	this.getColorRGBA = function() {
		if (this.cd_div === undefined) return curcolor;
		
		var huesat = getTransform(this.huehandle.getAttribute("transform"));
		var lum = getTransform(this.lumhandle.getAttribute("transform"))[1];
		var alpha = getTransform(this.alphahandle.getAttribute("transform"))[1];
		
		huesat[0] = huesat[0]/100.0;
		huesat[1] = 1.0 - huesat[1]/100.0;
		lum = 1.0 - lum/100.0;
		alpha = 1.0 - alpha/100.0;
		
		var rgb = HSL_To_RGB(huesat[0], huesat[1], 0.5);
		
		this.midlumstop.setAttribute("style", "stop-color:rgb(" + rgb[0] + "," + rgb[1] + "," + rgb[2] + ")");
		
		rgb = HSL_To_RGB(huesat[0], huesat[1], lum);
		
		this.fullalphastop.setAttribute("style", "stop-color:rgba(" + rgb[0] + "," + rgb[1] + "," + rgb[2] + ",1)");
		this.zeroalphastop.setAttribute("style", "stop-color:rgba(" + rgb[0] + "," + rgb[1] + "," + rgb[2] + ",0)");
		this.colorprev.setAttribute("fill", "rgba(" + rgb[0] + "," + rgb[1] + "," + rgb[2] + "," + alpha + ")")
		
		return curcolor = [rgb[0], rgb[1], rgb[2], alpha];
	}
	
	this.getColor = function() {return RGBA_To_CSS(this.getColorRGBA());}
	
	this.setColor = function(color) {
		var rgba = (typeof(color) == "string") ? CSS_To_RGBA(color) : color;
		if (this.cd_div === undefined) {curcolor = rgba; return;}
		
		var hsl = RGB_To_HSL(rgba[0], rgba[1], rgba[2]);
		
		console.log("hsl = " + hsl[0] + ", " + hsl[1] + ", " + hsl[2]);
		
		var rgb = HSL_To_RGB(hsl[0], hsl[1], 0.5);
		
		this.midlumstop.setAttribute("style", "stop-color:rgb(" + rgb[0] + "," + rgb[1] + "," + rgb[2] + ")");
		
		this.fullalphastop.setAttribute("style", "stop-color:rgba(" + rgba[0] + "," + rgba[1] + "," + rgba[2] + ",1)");
		this.zeroalphastop.setAttribute("style", "stop-color:rgba(" + rgba[0] + "," + rgba[1] + "," + rgba[2] + ",0)");
		this.colorprev.setAttribute("fill", RGBA_To_CSS(rgba))
		
		this.huehandle.setAttribute("transform", "translate(" + hsl[0]*100 + "," + (1.0-hsl[1])*100 + ")");
		this.lumhandle.setAttribute("transform", "translate(0," + (1.0-hsl[2])*100 + ")");
		this.alphahandle.setAttribute("transform", "translate(0," + (1.0-rgba[3])*100 + ")");
	}
	
	this.cd_div = undefined;
	
	this.show = function(parent_elem) {
		if (this.cd_div !== undefined) return;
		
		if (parent_elem !== undefined) this.parent = (typeof(parent_elem) == "string") ? document.getElementById(parent_elem) : parent_elem;
		
		var crect = this.parent.getBoundingClientRect();
		
		var cd_div = this.cd_div = document.createElement("div");
		cd_div.className = "colordialog";
		cd_div.style.width = "240px";
		cd_div.style.height = "240px";
		cd_div.style.paddingLeft = "10px";
		cd_div.style.paddingRight = "10px";
		cd_div.style.background = "rgba(0,0,255,0.5)";
		cd_div.style.position = "absolute";
		cd_div.style.top = (crect.bottom+window.scrollY) + "px";
		cd_div.style.left = (crect.left+window.scrollX) + "px";
		cd_div.style.zIndex = "1000";
		cd_div.onmousedown = cd_div.ontouchstart = function(e) {e.stopPropagation();};
		cd_div.onblur = function() {console.log("blur");that.hide();};
		cd_div.setAttribute("tabindex", "1");
		
		cd_div.innerHTML = SHEET_SVGCODE;
		
		this.svg = cd_div.getElementsByTagName("svg")[0];
		this.maing = cd_div.getElementsByClassName("cd_maing")[0];
		
		this.maing.onmousedown = mouseDown;
		this.maing.ontouchstart = function(e) {
			console.log("touchdown");
			mouseDown(e.touches[0]);
			e.preventDefault();
		};
		this.maing.ontouchmove = function(e) {
			mouseMove(e.touches[0]);
		};
		//this.maing.onmouseup = mouseUp;
		
		this.huehandle = cd_div.getElementsByClassName("cd_huehandle")[0];
		this.lumhandle = cd_div.getElementsByClassName("cd_lumhandle")[0];
		this.alphahandle = cd_div.getElementsByClassName("cd_alphahandle")[0];
		this.colorprev = cd_div.getElementsByClassName("cd_colorprev")[0];
		this.midlumstop = cd_div.getElementsByClassName("cd_midlumstop")[0];
		this.fullalphastop = cd_div.getElementsByClassName("cd_fullalphastop")[0];
		this.zeroalphastop = cd_div.getElementsByClassName("cd_zeroalphastop")[0];
		
		document.body.appendChild(cd_div);
		cd_div.focus();
		this.setColor(curcolor);
		this.onopen(RGBA_To_CSS(curcolor));
	}
	
	this.isShown = function() {return this.cd_div !== undefined;}
	
	this.hide = function() {
		if (this.cd_div === undefined) return;
		this.cd_div.remove();
		this.cd_div = undefined;
		this.onclose(RGBA_To_CSS(curcolor));
	}
	
	document.addEventListener("mousemove", mouseMove);
	document.addEventListener("mouseup", mouseUp);
}