// Library for complex number math in JavaScript.
//
// (c) Copyright Ice Fractal 2025 (www.icefractal.com)
//
// This code is open-source under the MIT License.
//
// You must include this copyright notice in anything that includes this code or modified parts of it.


/* EXAMPLE CODE:

var z = new Complex(3, 2); // Creates the number 3 + 2*i
var w = Exp(i.mul(Math.PI/3)); // Creates the number e^(i*pi/3)

var g = z.mul(w).div(i).sub(1); // Calculates z*w / i - 1

alert(g.toString());

// Result: 2.598076211353316 + i*0.23205080756887675

(JavaScript does not have operator overloading, so you must use these member functions for arithmetic.)

*/

// Refer to this article as a reference for complex math:
// https://icefractal.com/articles/complex-numbers/

// Let z = x + iy = re^(i(theta))

class Complex {
	constructor(x, y) {
		if (x instanceof Complex) {
			this.x = x.x;
			this.y = x.y;
		} else {
			this.x = +x || 0;
			this.y = +y || 0;
		}
	}
	
	// Creates a copy of this number.
	clone() {return new Complex(this);}
	
	// -z = -x - iy
	neg() {return new Complex(-this.x, -this.y);}
	
	//       _
	// 1/z = z / |z|^2
	inverse() {
		var msq = this.x*this.x + this.y*this.y;
		return new Complex(this.x/msq, -this.y/msq);
	}
	
	add(n) {if (!(n instanceof Complex)) n = new Complex(n); return new Complex(this.x + n.x, this.y + n.y);}
	sub(n) {if (!(n instanceof Complex)) n = new Complex(n); return new Complex(this.x - n.x, this.y - n.y);}
	mul(n) {if (!(n instanceof Complex)) return new Complex(this.x*n, this.y*n); return new Complex(this.x*n.x - this.y*n.y, this.x*n.y + this.y*n.x);}
	div(n) {if (!(n instanceof Complex)) return new Complex(this.x/n, this.y/n); return this.mul(n.inverse());}
	pow(n) {if (n === 2) return this.mul(this); return Exp(Log(this).mul(n));}
	abs()  {return Math.sqrt(this.x*this.x + this.y*this.y);}
	conj() {return new Complex(this.x, -this.y);}
	
	toString() {
		if (this.y == 0) {
			return this.x+"";
		} else {
			if (this.x == 0)
				return (this.y < 0) ? (this.y == -1 ? "-i" : ("-i*" + -this.y)) : (this.y == 1 ? "i" : ("i*" + this.y));
			else
				return (this.y < 0) ? (this.x + (this.y == -1 ? " - i" : (" - i*" + -this.y))) : (this.x + (this.y == 1 ? " + i" : (" + i*" + this.y)));
		}
	}
};

// Use a capital letter for complex functions, so they don't conflict with regular math. (It also implies that the principal branch is used, which it is.)

//                    _
// Complex Conjugate: z = x - iy
function Conj(n) {if (!n instanceof Complex) return n; return new Complex(n.x, -n.y);}

// Absolute Value: Abs(z) = sqrt(z.x*z.x + z.y*z.y)
function Abs(n) {if (!n instanceof Complex) return n < 0.0 ? -n : n; return Math.sqrt(n.x*n.x + n.y*n.y);}

// Argument: Arg(z) = theta, the polar angle wrapped to the range (-pi, pi]
function Arg(n) {if (!n instanceof Complex) return 0.0; return Math.atan2(n.y, n.x);}

// Real Part: Re(z) = x
function Re(n) {if (!(n instanceof Complex)) return n; return n.x;}

// Imaginary Part: Im(z) = y
function Im(n) {if (!(n instanceof Complex)) return 0; return n.y;}

// Exponential: Exp(z) = e^z
function Exp(n) { // e^(i*z) = e^x*cos(y) + i*e^x*sin(y)  where z = x + i*y
	if (!(n instanceof Complex)) return Math.exp(n);
	var r = n.x == 0.0 ? 1.0 : Math.exp(n.x);
	return new Complex(r*Math.cos(n.y), r*Math.sin(n.y));
}

// Power: Pow(z, w) = z^w
function Pow(a, b) {
	if (a === e) return Exp(b);
	if (!(a instanceof Complex)) {if (!(b instanceof Complex)) return Math.pow(a,b); a = new Complex(a);}
	return a.pow(b);
}

// Logarithm: Log(z, w) = ln(Abs(z)) + i*Arg(z)
function Log(n) {
	if (!(n instanceof Complex)) return Math.log(n);
	return new Complex(Math.log(Abs(n)), Arg(n));
}

// Sine: Sin(z) = (e^(i*z) - e^(-i*z)) / (2*i)
function Sin(n) {
	if (!(n instanceof Complex)) return Math.sin(n);
	var z = Exp(new Complex(-n.y, n.x)).sub(Exp(new Complex(n.y, -n.x)));
	return new Complex(z.y*0.5, -z.x*0.5);
}

// Cosine: Cos(z) = (e^(i*z) + e^(-i*z)) / 2
function Cos(n) {
	if (!(n instanceof Complex)) return Math.cos(n);
	var z = Exp(new Complex(-n.y, n.x)).add(Exp(new Complex(n.y, -n.x)));
	return new Complex(z.x*0.5, z.y*0.5);
}

// Lanczos approximation for the gamma function.
function Gamma(z) {
	const g = 5;
	const p = [ // Series Coefficients
		1.0000000001900148240,
		76.180091729471463483,
		-86.505320329416767653,
		24.014098240830910488,
		-1.2317395724501553873,
		0.001208650973866178506,
		-5.3952393849531283786e-6
	];
	const M_ROOT2PI = 2.50662827463100050241576528481105;
	
	if (!(z instanceof Complex)) z = new Complex(z);
	
	if (z.x < 0.5) { // Use Euler's relfection formula for Re(z) < 0.5
		return (Sin(z.mul(Math.PI)).mul(Gamma(new Complex(1.0 - z.x, z.y)))).inverse().mul(Math.PI);
	} else {
		z = z.sub(1);
		var s = p[0];
		for (var n = 1; n < p.length; n++) {
			s = new Complex(z.x+n, z.y).inverse().mul(p[n]).add(s);
		}
		var t = new Complex(z.x + g + 0.5, z.y);
		return t.pow(z.add(0.5)).mul(Exp(t.neg())).mul(s).mul(M_ROOT2PI);
	}
}

// Define the constant i. (Be careful: if you use i as a local variable, it will shadow this. It is the convention in C to use I instead of i. Both are provided here.)
var i = new Complex(0, 1);
var I = i;

